# 📗 DEV_BRIEF — MappingDB_v1

*(Document technique — destiné aux IA développeuses : Codex / AI Studio)*

---

## 1. Résumé technique

Le module `MappingDB_v1` doit implémenter une base SQLite nommée `synomap.db`, destinée à remplacer le fichier `mapping_entries.txt`.
Il fournit une API Python claire, robuste et extensible permettant :

* d’enregistrer des **événements Sonarr/Radarr** (mapping brut),
* de consolider l’état courant d’un mapping pour chaque torrent (table `mapping_latest`),
* de détecter les anomalies (missing, multi-candidats, incohérences),
* de fournir une **structure compatible A4** (partielle) incluant Identity, Paths, Diagnostic, State minimal.

Cette base est la **Source-of-Truth** pour tous les modules Synomap V2.

---

## 2. Contraintes techniques

* **Langage** : Python 3.10+
* **Stockage** : SQLite (fichier local `synomap.db`)
* **Pas de dépendances externes** autres que `sqlite3` (lib standard).
* **Robustesse > performance**
* **Code clair, commenté**, pas de magie, pas d'abstractions inutiles.
* **Gestion stricte des erreurs** : exceptions explicites et logs JSON.
* **APIs stables** pour futur usage dans B1/B3/B4.
* **Migration legacy** (`mapping_entries.txt`) doit être idempotente.
* **Aucune action FS** (pas de hardlink, pas de copie, pas de suppression).
* **Aucune interaction QB API** (réservé au Checker).

---

## 3. Signature / API attendue

Le module doit exposer une **classe principale** :

```python
class MappingStore:
    def __init__(self, db_path: str = "synomap.db"):
        ...

    def insert_event(self, event: dict) -> None:
        """Insert a raw mapping event (from Sonarr/Radarr)."""

    def get_mapping(self, infohash: str) -> dict | None:
        """Return consolidated mapping for this infohash."""

    def import_legacy_file(self, filepath: str) -> None:
        """Migrate old mapping_entries.txt into SQLite."""

    def list_events(self, infohash: str) -> list[dict]:
        """Return raw events for analysis/debug."""

    def compute_diagnostic(self, events: list[dict]) -> dict:
        """Return diagnostic: OK/MISSING/MULTI/PARTIAL/CORRUPT."""

    def rebuild_latest(self, infohash: str) -> None:
        """Recompute mapping_latest entry after new events."""
```

Optionnel :
Méthodes internes `_init_db()`, `_consolidate()`, etc.

---

## 4. Données d’entrée (format strict)

### 4.1. Événements Sonarr / Radarr

Chaque événement inséré via `insert_event()` doit suivre le format :

```json
{
  "infohash": "ABC123...",
  "source": "/path/to/torrent_source/",
  "destination": "/path/to/final_folder/",
  "type": "tv" | "movie",
  "timestamp": "2025-11-28T18:12:34Z",
  "release_group": "FGT",
  "files": ["file1.mkv", "file2.srt"]
}
```

Champs obligatoires :

* infohash
* source
* destination
* type
* timestamp

Champs optionnels :

* release_group
* files (peut être vide ou absent)

### 4.2. Legacy (mapping_entries.txt)

Format approximatif (à adapter à ton échantillon réel) :

```
<INFOHASH>|<SRC_PATH>|<DEST_PATH>|<TYPE>|<TIMESTAMP>
```

Le module doit être capable :

* de parser chaque ligne,
* de convertir vers un événement `insert_event()`,
* de gérer doublons + ambiguïtés.

---

## 5. Données de sortie (format strict)

Appel : `get_mapping(infohash: str)`
Retour attendu :

```json
{
  "infohash": "ABC123...",
  "source_path": "/data/.../",
  "dest_path": "/syno/...",
  "type": "tv",
  "events": [...],
  "diagnostic": {
      "status": "OK" | "MISSING" | "MULTI" | "PARTIAL" | "CORRUPT",
      "detail": "string description",
      "candidates": ["..."]
  }
}
```

Si aucun mapping :
→ retour `None`.

Si diagnostic = `MISSING`, retour :

```json
{
  "infohash": "ABC123...",
  "diagnostic": { "status": "MISSING" }
}
```

---

## 6. Logique à implémenter (étapes)

### 6.1. Initialisation de la DB

* Vérifier si DB existe, sinon créer fichier + tables :

  * `mapping_events`
  * `mapping_latest`
  * `mapping_diagnostics` (optionnel)

### 6.2. Insertion d’un événement

* Valider format de l’événement.
* Insérer dans `mapping_events`.
* Recalculer mapping_latest pour cet infohash.

### 6.3. Consolidation (mapping_latest)

Pour un infohash donné :

1. Récupérer tous les événements (ordonnés par timestamp).
2. Détecter :

   * aucun événement → diagnostic `MISSING`
   * plusieurs dest_path différents → diagnostic `MULTI`
   * dest_path invalide (None / vide) → `PARTIAL`
3. Déterminer le “mapping consolidé” :

   * dernier événement cohérent
   * dest_path stable
4. Stocker résultat dans `mapping_latest`.

### 6.4. Diagnostic

La fonction `compute_diagnostic(events)` doit produire :

| Condition                           | Diagnostic |
| ----------------------------------- | ---------- |
| 0 événements                        | MISSING    |
| Plusieurs dest différents           | MULTI      |
| Un unique événement mais incomplet  | PARTIAL    |
| Tous champs valides, unique dest    | OK         |
| Données incohérentes syntaxiquement | CORRUPT    |

### 6.5. Import legacy

* Lire chaque ligne du fichier TXT.
* Parser les champs.
* Convertir vers un événement identique à Radarr/Sonarr.
* Insérer via `insert_event()`.
* Pas de duplication :

  * si un événement identique existe → ignorer.

---

## 7. Cas limites

* **Multi-candidats**
  Plusieurs événements, dest_path différents → état MULTI.

* **Ligne legacy corrompue**
  → log ERROR, ignorer l’entrée.

* **Événement incomplet**
  → insert_event() doit stocker l’événement, mais diagnostiquer PARTIAL.

* **Type incohérent**
  Type non {tv, movie} → flag INVALID mais pas crash.

* **DB verrouillée**
  Timeout SQLite → remonter une exception propre (“DB_LOCKED”).

* **Conflit hash (cas rare)**
  2 événements avec même infohash mais type différent → MULTI + flag TYPE_CONFLICT.

---

## 8. Critères qualité obligatoires

* Code Python lisible, clair, commenté.
* Pas de dépendances externes.
* Fonctions pures autant que possible.
* Aucune logique Synomap B1/B3/B4 dans ce module.
* Jamais de silent fail.
* Toutes les anomalies doivent être détectées, loguées, retournées.
* Structure strictement compatible A4 (Identity + Paths + Diagnostic).
* Préparer le terrain pour l’ajout futur :

  * hash fichier,
  * flags de décision,
  * replays B3/B4.

---

## 9. Cas de tests conceptuels

### Test 1 — Nominal Sonarr

Entrée : un événement unique complet.
Sortie : diagnostic = OK.

### Test 2 — Missing

Entrée : infohash non présent.
Sortie : diagnostic = MISSING.

### Test 3 — Multi

Entrée : deux événements avec dest différents.
Sortie : diagnostic = MULTI + candidates listées.

### Test 4 — Legacy ambigu

Entrée : TXT contenant deux chemins différents.
Sortie : diagnostic = MULTI.

### Test 5 — Partial

Entrée : événement sans destination.
Sortie : diagnostic = PARTIAL.

### Test 6 — Type conflict

event1 = type tv, event2 = type movie
→ diagnostic = MULTI + flag TYPE_CONFLICT.

---

## 10. Livrables attendus

* Un fichier Python `mapping_store.py` contenant :

  * la classe `MappingStore` complète,
  * les méthodes spécifiées,
  * un docstring en tête expliquant le rôle du module.

* Un fichier SQLite `synomap.db` créé automatiquement à l’exécution (si inexistant).

* Un block de tests unitaires simples (optionnel mais apprécié) :

  * ou au moins un script de démonstration.

* Aucune dépendance, aucun placeholder.

---

